'use strict';

/* --------------------------------------------------------------
 switcher.js 2017-11-21
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2017 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Switcher Widget
 *
 * This widget originates from the "switcher" mode of the existing checkbox widget. Because of the increased
 * complexity of the old widget code, the switcher mode is now server by this file. Apply the widget in a parent
 * container and it will search and convert all the checkbox/radio instances into switchers.
 *
 * ### Options
 *
 * **On State | `data-switcher-on-state` | String | Optional**
 *
 * Define the content of the "on" state.
 *
 * **Off State | `data-switcher-off-state` | String | Optional**
 *
 * Define the content of the "off" state.
 *
 * **Selector | `data-switcher-selector` | String | Optional**
 *
 * Set the selector of the checkboxes to be converted to switcher instances. It defaults to **input:checkbox**.
 *
 * ### Methods
 *
 * **Checked**
 *
 * ```js
 * // Set the checked value of the single checkbox selection (no change event will be triggered!).
 * $('table input:checkbox').switcher('checked', true);
 * ```
 *
 * ### Examples
 *
 * In the following example the checkbox element will be converted into a single-checkbox instance.
 *
 * ```html
 * <div class="wrapper" data-gx-widget="switcher">
 *   <input type="checkbox" />
 * </div>
 * ```
 *
 * @todo Add method for disabling the switcher widget (e.g. $('#my-switcher').switcher('disabled', true));
 *
 * @module Admin/Widgets/switcher
 */
gx.widgets.module('switcher', [], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Default Options
     *
     * @type {Object}
     */
    var defaults = {
        onState: '<span class="fa fa-check"></span>',
        offState: '<span class="fa fa-times"></span>',
        selector: 'input:checkbox'
    };

    /**
     * Final Options
     *
     * @type {Object}
     */
    var options = $.extend(true, {}, defaults, data);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {};

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Set the "checked" property of the single checkbox instances.
     *
     * This method will update the value and display of the widgets without triggering a "change" event.
     *
     * @param {Boolean} isChecked The checkbox values will be updated along with their representation.
     *
     * @return {jQuery} Returns the jQuery selector for chained calls.
     */
    function _checked(isChecked) {
        $(this).prop('checked', isChecked);
        _onCheckboxChange.call(this);
        return $(this);
    }

    /**
     * Add Public Module Methods
     *
     * Example: $('input:checkbox').switcher('checked', false);
     */
    function _addPublicMethod() {
        if ($.fn.switcher) {
            return; // Method is already registered.
        }

        $.fn.extend({
            switcher: function switcher(action) {
                switch (action) {
                    case 'checked':
                        for (var _len = arguments.length, args = Array(_len > 1 ? _len - 1 : 0), _key = 1; _key < _len; _key++) {
                            args[_key - 1] = arguments[_key];
                        }

                        return _checked.apply(this, args);
                }
            }
        });
    }

    /**
     * On Switcher Click Event
     *
     * Delegate the click event to the checkbox elements which will update the DOM accordingly.
     *
     * @param {object} event
     */
    function _onSwitcherClick(event) {
        event.stopPropagation();

        if ($(this).hasClass('disabled')) {
            return false; // The switcher is disabled.
        }

        var $checkbox = $(this).find('input:checkbox');

        $checkbox.prop('checked', !$checkbox.prop('checked')).trigger('change');
    }

    /**
     * On Checkbox Change
     *
     * This callback will update the display of the widget. It will perform the required animations and set the
     * respective state classes.
     */
    function _onCheckboxChange() {
        var $checkbox = $(this);
        var $switcher = $checkbox.parent();

        if (!$switcher.hasClass('checked') && $checkbox.prop('checked')) {
            $switcher.addClass('checked');
        } else if ($switcher.hasClass('checked') && !$checkbox.prop('checked')) {
            $switcher.removeClass('checked');
        }
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        _addPublicMethod();
        $this.find(options.selector).each(function () {
            var $checkbox = $(this);
            var title = $checkbox.prop('title') ? 'title="' + $checkbox.prop('title') + '"' : '';

            $checkbox.wrap('<div class="switcher" ' + title + '></div>').parent().append('\n\t\t\t\t\t<div class="switcher-toggler"></div>\n\t\t\t\t\t<div class="switcher-inner">\n\t\t\t\t\t\t<div class="switcher-state-on">' + options.onState + '</div>\n\t\t\t\t\t\t<div class="switcher-state-off">' + options.offState + '</div>\n\t\t\t\t\t</div>\n\t\t\t\t');

            // Bind the switcher event handlers.
            $checkbox.parent().on('click', _onSwitcherClick).on('change', 'input:checkbox', _onCheckboxChange);

            // Trigger the change event to update the checkbox display.
            _onCheckboxChange.call($checkbox[0]);
        });

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
